<?php
// File: application/models/Transaction_model.php
// Model yang menangani transaksi.

class Transaction_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }
    
    /**
     * Mendapatkan produk berdasarkan ID.
     * @param int $product_id
     * @return object
     */
    public function get_product_by_id($product_id) {
        return $this->db->get_where('products', ['id' => $product_id])->row();
    }
    
    /**
     * Mendapatkan afiliasi berdasarkan kode referral.
     * @param string $referral_code
     * @return object
     */
    public function get_affiliate_by_code($referral_code) {
        return $this->db->get_where('affiliates', ['referral_code' => $referral_code])->row();
    }

    /**
     * Mencatat transaksi dan membuat entri jurnal.
     * @param array $transaction_data
     * @return bool
     */
    public function process_checkout($transaction_data) {
        // Mulai transaksi database untuk memastikan konsistensi
        $this->db->trans_start();

        // 1. Catat transaksi
        $this->db->insert('transactions', $transaction_data);
        $transaction_id = $this->db->insert_id();

        // 2. Buat entri jurnal ganda
        // Jurnal untuk penjualan: Debit Kas, Kredit Pendapatan Penjualan
        $journal_sale = [
            'transaction_id' => $transaction_id,
            'date' => date('Y-m-d'),
            'description' => 'Penjualan produk ID ' . $transaction_data['product_id'],
            'debit_account_id' => 1, // ID Akun Kas
            'credit_account_id' => 5, // ID Akun Pendapatan Penjualan
            'amount' => $transaction_data['amount']
        ];
        $this->db->insert('journals', $journal_sale);

        // Jika ada komisi afiliasi, buat entri jurnal tambahan
        if ($transaction_data['affiliate_id']) {
            // Jurnal untuk beban komisi: Debit Beban Komisi, Kredit Hutang Komisi Afiliasi
            $journal_commission = [
                'transaction_id' => $transaction_id,
                'date' => date('Y-m-d'),
                'description' => 'Beban komisi afiliasi ID ' . $transaction_data['affiliate_id'],
                'debit_account_id' => 6, // ID Akun Beban Komisi Afiliasi
                'credit_account_id' => 3, // ID Akun Hutang Komisi Afiliasi
                'amount' => $transaction_data['commission_amount']
            ];
            $this->db->insert('journals', $journal_commission);
        }

        // Akhiri transaksi database
        $this->db->trans_complete();

        return $this->db->trans_status();
    }
    
    /**
     * Mendapatkan transaksi terbaru untuk Dasbor Merchant.
     * @param int $user_id
     * @return array
     */
    public function get_recent_transactions($user_id) {
        $this->db->select('t.*, u.username as customer_name, p.nama_barang as product_name');
        $this->db->from('transactions t');
        $this->db->join('tbl_user u', 'u.id_user = t.affiliate_id');
        $this->db->join('tbl_barang p', 'p.id_barang = t.business_id');
//        $this->db->join('merchants m', 'm.id = t.merchant_id');
        $this->db->where('u.id_user', $user_id);
        $this->db->order_by('t.transaction_date', 'DESC');
        $this->db->limit(10);
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Mendapatkan laporan penjualan per produk untuk Dasbor Merchant.
     * @param int $user_id
     * @return array
     */
    public function get_sales_by_product($user_id) {
        $this->db->select('p.nama_barang, COUNT(t.id) as total_sales, SUM(t.amount) as total_revenue');
        $this->db->from('transactions t');
        $this->db->join('tbl_barang p', 'p.id_barang = t.business_id');
        $this->db->join('affiliates m', 'm.id = t.business_id');
        $this->db->where('m.business_id', $user_id);
        $this->db->group_by('p.id_barang, p.nama_barang');
        $this->db->order_by('total_revenue', 'DESC');
        $query = $this->db->get();
        return $query->result_array();
    }
}

