<?php
// File: application/controllers/Products.php
// Controller untuk manajemen produk dan daftar produk.

class Products extends MY_Controller {

    public function __construct() {
        parent::__construct();
        $this->check_permission('view_dashboard');
        $this->load->model('Product_model');
        $this->load->helper('form');
        $this->load->library('form_validation');
    }

    public function index() {
        $data['title'] = "Daftar Produk";
        $data['products'] = $this->db->get('products')->result_array();
        
        $this->load->view('templates/header', $data);
        $this->load->view('products/index', $data);
        $this->load->view('templates/footer', $data);
    }
    
    /**
     * Halaman manajemen produk untuk merchant.
     */
    public function manage() {
        $this->check_permission('manage_products');
        
        $merchant = $this->db->get_where('merchants', ['user_id' => $this->session->userdata('user_id')])->row();
        
        $data['title'] = "Manajemen Produk Saya";
        $data['products'] = $this->Product_model->get_products_by_merchant($merchant->id);
        
        $this->load->view('templates/header', $data);
        $this->load->view('products/manage', $data);
        $this->load->view('templates/footer', $data);
    }

    /**
     * Menambah atau mengedit produk.
     */
    public function form($product_id = NULL) {
        $this->check_permission('manage_products');
        
        $data['product'] = null;
        if ($product_id) {
            $data['product'] = $this->Product_model->get_product($product_id);
            if (!$data['product']) {
                show_404();
            }
        }
        
        $data['title'] = $product_id ? "Edit Produk" : "Tambah Produk Baru";

        $this->load->view('templates/header', $data);
        $this->load->view('products/form', $data);
        $this->load->view('templates/footer', $data);
    }

    /**
     * Memproses form tambah/edit produk.
     */
    public function save() {
        $this->check_permission('manage_products');

        $this->form_validation->set_rules('name', 'Nama Produk', 'trim|required|max_length[255]');
        $this->form_validation->set_rules('description', 'Deskripsi', 'trim');
        $this->form_validation->set_rules('price', 'Harga', 'trim|required|numeric|greater_than[0]');
        $this->form_validation->set_rules('stock', 'Stok', 'trim|required|integer|greater_than_equal_to[0]');
        $this->form_validation->set_rules('affiliate_commission_rate', 'Rate Komisi Afiliasi', 'trim|required|numeric|greater_than_equal_to[0]');

        $product_id = $this->input->post('product_id', TRUE);
        
        if ($this->form_validation->run() === FALSE) {
            $this->form($product_id);
        } else {
            $merchant = $this->db->get_where('merchants', ['user_id' => $this->session->userdata('user_id')])->row();
            
            $data = [
                'merchant_id' => $merchant->id,
                'name' => $this->input->post('name', TRUE),
                'description' => $this->input->post('description', TRUE),
                'price' => $this->input->post('price', TRUE),
                'stock' => $this->input->post('stock', TRUE),
                'affiliate_commission_rate' => $this->input->post('affiliate_commission_rate', TRUE)
            ];

            if ($product_id) {
                $this->Product_model->update_product($product_id, $data);
                $this->session->set_flashdata('success', 'Produk berhasil diperbarui.');
            } else {
                $this->Product_model->add_product($data);
                $this->session->set_flashdata('success', 'Produk berhasil ditambahkan.');
            }
            redirect('products/manage');
        }
    }

    /**
     * Menghapus produk.
     */
    public function delete($product_id) {
        $this->check_permission('manage_products');

        // Pastikan produk milik merchant yang sedang login
        $merchant = $this->db->get_where('merchants', ['user_id' => $this->session->userdata('user_id')])->row();
        $product = $this->db->get_where('products', ['id' => $product_id, 'merchant_id' => $merchant->id])->row();
        
        if (!$product) {
            show_error('Anda tidak memiliki izin untuk menghapus produk ini.', 403);
            exit;
        }

        $this->Product_model->delete_product($product_id);
        $this->session->set_flashdata('success', 'Produk berhasil dihapus.');
        redirect('products/manage');
    }
}
